// ****************
// gpm.h file
// *****************
/*
Gpm

License: GNU General Public License

Cordier Yves 
V1.0  : 23/09/2025
v1.1  : 24/09/2025
        including various versions of the shield 
		GPAM, GPMS, GPMP
		

this librarie is suitable to several versions of DPM shield
 a: The first DS-GPM shield 
 b: The extend DS-GPAM shield
 c: The actual DS-GPM+ shield
 
 a : This shield has in addition 4 I/O 
     which could be positioned in digital Output | digital Input | anolog Input (8bits)
 b : This shield has in addition 4 I/O 
     which could be positioned in digital Output | digital Input | anolog Input (8bits)
	 In addition a3 axis accelerometer 
 c : This shield has only a 3 axis accelerometer
 
 However all functions discribe un this librarie gives a value with all versions.
 If they supposed to manage with a non existent feature, it returs 0
 
 You must declear the version you have
 before including the librarie in your sketch
 
 a : #define _GPAM  // correpond at IES-Shield-GPAM
 b : #define _GPM  // correspond at DS-GPM Shield
 c : #define _GPMP // correspond at sHLD-GPM+
 exemple :
 
 //#define _GPAM // uncomment this line for GPMA version correpond at IES-Shield-GPAM
 //#define _GPM  // uncomment this line for GPM version correspond at DS-GPM Shield
 //#define _GPMP // uncomment this line for GPMP version correspond at sHLD-GPM+
 
 #include <gpm.h>
 
 note that if no version is specified, the basics functions of gps 
 will be operationals
 
	
*/
/******************************************************************************
* Includes
******************************************************************************/
#include "gpm.h"
#include <Wire.h>

//***************************************************************************
// Les constructeurs
// avec l'adresse par defaut 0x68
Gpm::gpm() {
	init();
}
// avec l'adresse a
Gpm::gpm(byte a) {
	setAdress(a);
	init();
}

//***************************************************************************
// manipulation de l'adresse
void Gpm::setAdress(byte a) {
	GPM = a;
}

//***************************************************************************
// initialisation de la communication
void Gpm::init() {
	Wire.begin();
}

//***************************************************************************
// definition des modes d'entrée / Sortie similaire à pinMode Arduino
// Ici le mode INPUT ey INPUT_PULLUP sont traités de manière identique
// Il appartient à l'utilisateur d'installer une résistance de pullup (10k) entre la
// pin d'entrée du module et la borne vcc
void Gpm::gpmIOpinMode(byte p, byte mode) {
#ifdef _IO
	// préparation des valeurs internes à la classe
	byte masque = 0x01<<(p-1);
	if ((mode == INPUT) || (mode == INPUT_PULLUP)) {
		masqueInputDirection |= masque;
		masqueIoType &= ~masque;
	} else if (mode == OUTPUT) {
		masqueInputDirection &= ~masque;
		masqueIoType &= ~masque;
	} else if (mode == ANALOG) {
		masqueInputDirection &= ~masque;
		masqueIoType |= masque;
	}
	// mise à jour du module GPM
	Wire.beginTransmission(GPM);
	Wire.write(0);  // Send register start address
	Wire.write(masqueInputDirection);
	Wire.write(masqueIoType);
	Wire.endTransmission();
#endif
}
// nombre de satellites scannés
int Gpm::nbSateliteFound() {
	return getDouble(34);
}

//***************************************************************************
// Y a-t-il assez de satellites (>2)
boolean Gpm::sateliteFound() {
	return (nbSateliteFound() > 2);
}

//***************************************************************************
// lecture en bloc des entrées digitales
byte Gpm::gpmIOdigitalReadBlock() {
#ifdef _IO
	byte V = getSingle(_IO);
	return (V);
#else
	return 0;
#endif
}

//***************************************************************************
// lecture d'une entrée digitale (numero n 1..4)
boolean Gpm::gpmIOdigitalRead(byte n) {
#ifdef _IO
	return ((gpmIOdigitalReadBlock() & (0x01 << (n - 1))) > 0);
#else
	return false;
#endif
}


//***************************************************************************
// lecture d'une entrée analogique (numéro n 1..4)
byte Gpm::gpmIOanalogRead(byte n) {
#ifdef _ANA0
	return getSingle(_ANA0 - 1 + n);
#else
	return 0;
#endif
}


//***************************************************************************
// écriture d'une sortie digitale (numero n 1..4 , true | false
void Gpm::gpmIOdigitalWrite(byte n, boolean v) {
#ifdef _IO
	byte vact = gpmIOdigitalReadBlock();
	byte masque = 0x01 << (n - 1);
	if (v) vact |= masque;
	else vact &= ~masque;
	Wire.beginTransmission(GPM);
	Wire.write(2);  // Send register start address
	Wire.write(vact);
	Wire.endTransmission();
#endif
}

//***************************************************************************
// renvoi l'heure en nb de secondes depuis minuit
uint32_t Gpm::getTime() {
	// Point to Hours register
	uint32_t Data = uint32_t(getDouble(0)) * 3600;  // Read registers from GPM
	// Point to Minutes register
	Data += uint32_t(getDouble(2)) * 60;  // Read registers from GPM
	// Point to Seconds register
	Data += uint32_t(getDouble(4));  // Read registers from GPM
	return Data;
}

//***************************************************************************
// renvoi le jour du mois 1..31
uint8_t Gpm::getDay() {
	return getDouble(6);
}

//***************************************************************************
// renvoi le mois 1..12
uint8_t Gpm::getMonth() {
	return getDouble(8);
}

//***************************************************************************
// renvoi l'année ex : 2025
uint16_t Gpm::getYear() {
	return uint8_t(getMultiple(10, 4));
}

//***************************************************************************
// renvoi l'altitude en m
long Gpm::getAltitude() {

	return getMultiple(39, 5);
}

//***************************************************************************
// renvoi la latitude en unité de 1E-6 degrés
long Gpm::getLatitude() {
#define _1E6 (1000000)
	long val = 0;
	val = getDouble(14) * _1E6;
	val += (float(getSingle(16)) * (1. / 60)) * _1E6;
	val += (float(getSingle(17)) * (1. / 600)) * _1E6;
	val += (float(getSingle(18)) * (1. / 6000)) * _1E6;
	val += (float(getSingle(19)) * (1. / 60000)) * _1E6;
	val += (float(getSingle(20)) * (1. / 600000)) * _1E6;
	val += (float(getSingle(21)) * (1. / 6000000)) * _1E6;
	if (getSingle(22) != 'N') val = -val;
	return val;
}

//***************************************************************************
// renvoi la longitude en unité de 1E-6 degrés
long Gpm::getLongitude() {
#define _1E6 (1000000)
	long val = 0;
	val = getTriple(23) * _1E6;
	val += (float(getSingle(26)) * (1. / 60)) * _1E6;
	val += (float(getSingle(27)) * (1. / 600)) * _1E6;
	val += (float(getSingle(28)) * (1. / 6000)) * _1E6;
	val += (float(getSingle(29)) * (1. / 60000)) * _1E6;
	val += (float(getSingle(30)) * (1. / 600000)) * _1E6;
	val += (float(getSingle(31)) * (1. / 6000000)) * _1E6;
	if (getSingle(32) != 'E') val = -val;
	return val;
}

//***************************************************************************
// renvoi la vitesse en unité de 0.1 km/s
int Gpm::getSpeed() {
	return int(getMultiple(52, 4));
}

//***************************************************************************
// renvoi la direction de deplacement en unité de 1/10 degré /nord geographique
int Gpm::getHeadingGeo() {
	return int(getMultiple(44, 4));
}

//***************************************************************************
// renvoi la direction de deplacement en unité de 1/10 degré /nord magnetique
int Gpm::getHeadingMag() {
	return int(getMultiple(48, 4));
}

//***************************************************************************
// renvoi la valeur de l'accelerometre en X (>0 ou <0)
int Gpm::getAccelX()
{  
#ifdef _AX
	return getSignedWord(_AX);	
#else
	return 0;
#endif
}

//***************************************************************************
// renvoi la valeur de l'accelerometre en X (>0 ou <0)
int Gpm::getAccelY()
{ 
#ifdef _AY
	return getSignedWord(_AY);	
#else
	return 0;
#endif
}

//***************************************************************************
// renvoi la valeur de l'accelerometre en X (>0 ou <0)
int Gpm::getAccelZ()
{ 
#ifdef _AZ
	return getSignedWord(_AZ);	
#else
	return 0;
#endif
}
//***************************************************************************
// modifie la référence de l'accélération sur X
void Gpm::setRefGx(int v) {
#ifdef _AX
	v1Gx=v;
#endif
}

//***************************************************************************
// modifie la référence de l'accélération sur Y
void Gpm::setRefGy(int v) {
#ifdef _AY
	v1Gy=v;
#endif
}

//***************************************************************************
// modifie la référence de l'accélération sur Z
void Gpm::setRefGz(int v) {
#ifdef _AZ
	v1Gz=v;
#endif
}

//***************************************************************************
// renvoi le nombre de G sur l'axe X (relativement à 1Gx)
float Gpm::getGx() {
#ifdef _AX
	if ( v1Gx!=0)return float(getAccelX())/v1Gx;
	else 
#endif
	return 0;}

//***************************************************************************
// renvoi le nombre de G sur l'axe Y (relativement à 1Gy)
float Gpm::getGy() {
#ifdef _AY
	if (v1Gy!=0) return float(getAccelY())/v1Gy;
	else 
#endif	
	return 0;}

//***************************************************************************
// renvoi le nombre de G sur l'axe Z (relativement à 1Gz)
float Gpm::getGz() {
#ifdef _AZ
	if (viGz!=0) return float(getAccelZ())/v1Gz;
	else 
#endif
	return 0;}

//***************************************************************************
// renvoi l'angle de tangage en degrés
int Gpm::getPitch() {
#ifdef _PITCH
	return int(getSignedByte(_PITCH));
#else
	return 0;
#endif
}

//***************************************************************************
// renvoi l'angle de roulis en degrés
int Gpm::getRoll() {
#ifdef _ROLL
	return int(getSignedByte(_ROLL));
#else
	return 0;
#endif
}


//***************************************************************************
// recupere n bytes de l'I2c à partir de l'adresse addr
uint32_t Gpm::getMultiple(byte addr, byte n) {

	uint32_t value = 0;

	byte C = 0;

	Wire.beginTransmission(GPM);
	Wire.write(addr);  // Send register start address
	Wire.endTransmission();

	Wire.requestFrom(GPM, n);         // Request n registers
	while (Wire.available() < n) {};  // Wait for n byte
	for (int i = 0; i < n; i++) { value = 10 * value + Wire.read(); } // adjust fo an unsigned long
	return (value);
}

//***************************************************************************
// recupere 3 bytes de l'I2c à partir de l'adresse addr
int Gpm::getTriple(byte addr) {

	int value = 0;
	byte HH_Byte = 0;
	byte H_Byte = 0;
	byte L_Byte = 0;

	Wire.beginTransmission(GPM);
	Wire.write(addr);  // Send register start address
	Wire.endTransmission();

	Wire.requestFrom(GPM, 3);         // Request triple register
	while (Wire.available() < 3) {};  // Wait for triple byte
	HH_Byte = Wire.read(); // Store first
	H_Byte = Wire.read();  // Store second
	L_Byte = Wire.read();  // Store third
	
	value = (int(HH_Byte) * 100) + (H_Byte * 10) + L_Byte;  // Adjust for one int
	return (value);
}


//***************************************************************************
// recupere 2 bytes de l'I2c à partir de l'adresse addr
int Gpm::getDouble(byte addr) {

	int value = 0;
	byte H_Byte = 0;
	byte L_Byte = 0;

	Wire.beginTransmission(GPM);
	Wire.write(addr);  // Send register start address
	Wire.endTransmission();

	Wire.requestFrom(GPM, 2);         // Request double register
	while (Wire.available() < 2) {};  // Wait for double byte
	H_Byte = Wire.read();             // Store first
	L_Byte = Wire.read();             // Store second
	value = (H_Byte * 10) + L_Byte;   // Adjust for one int
	return (value);
}


//***************************************************************************
// recupere 1 byte de l'I2c à partir de l'adresse addr
byte Gpm::getSingle(int addr) {

	byte value = 0;
	Wire.beginTransmission(GPM);
	Wire.write(addr);  // Send register start address
	Wire.endTransmission();

	Wire.requestFrom(GPM, 1);         // Request register
	while (Wire.available() < 1) {};  // Wait for single byte
	value = Wire.read();              // and store
	return (value);
}

//***************************************************************************
// recupere 1 byte signé de l'I2c à partir de l'adresse addr
int Gpm::getSignedByte(int addr) {

	byte value = getSingle(addr);
	if ((value & 0x80)>0) return (-(int(value & 0x7F)));
	else return (int(value));
}

//***************************************************************************
// recupere un int (2 byte signé) de l'I2c à partir de l'adresse addr
int Gpm::getSignedWord(int addr) {

	word valueb = getSingle(addr);
	word valueh = word(getSingle(addr+1))<<8;	
	Serial.println();
	Serial.print(valueb,BIN);
	Serial.print(" ");
	Serial.println(valueh,BIN);
	return (int(valueb | valueh));
}
